/*
MB1_DIDO: Digitial IO tool for MB1

Copyright (c) 2019, 2020, MiTAC Computing Technology Corporation ("MCT").
All rights reserved. 

Redistribution and use in source and binary forms, with or without 
modification, are permitted provided that the following conditions are met:

  * Redistributions of source code must retain the above copyright
    notice, this list of conditions and the following disclaimer.
  * Redistributions in binary form must reproduce the above copyright
    notice, this list of conditions and the following disclaimer in the
    documentation and/or other materials provided with the distribution.
  * Neither the name of the author nor the names of its contributors may
    be used to endorse or promote products derived from this software
    without specific prior written permission.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.

MCT shall at all times retain ownership of the Software as originally 
downloaded by you and all subsequent downloads of the Software by you. The 
Software (and the copyright, and other intellectual property rights of 
whatever nature in the Software, including any modifications made thereto) 
are and shall remain the property of MCT.

MCT reserves the right to grant licences to use the Software to third parties.

*/

#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/io.h>


#define APP_NAME     "MB1 DIDO Utility"
#define APP_VERSION  "1.00"

#define MSG_OK             0
#define MSG_InvalidParam   1
#define MSG_NOTSUPPORTED   2
#define MSG_PERMITDENY     3
#define MSG_HELP           99

#define UINT8  unsigned char
#define inportb(a) (inb(a))
#define outportb(a,b) (outb(b,a))

//-- 0x2E or 0x4E for SIO_BASE , depend on hardware strapping pin '2E_4E_SEL'
//  In MB1, it uses the '0x2E'.
#define SIO_BASE            0x2E
#define	SIO_INDEX 			(SIO_BASE)
#define	SIO_DATA 			(SIO_BASE + 1) 
#define TOTAL_PINS          4

enum  { 
  DIO_GPO1 = 01,
  DIO_GPO2,
  DIO_GPO3,
  DIO_GPO4,
  DIO_GPI1 = 11,  
  DIO_GPI2,  
  DIO_GPI3,  
  DIO_GPI4  
}; 

typedef struct {
UINT8 LD;       //Logic device
UINT8 GBIT;     //Group bit
UINT8 REG_DIR;
UINT8 REG_DAT;
UINT8 BIT;      //pin bit
} GPIO_INFO ;	

//---- Global variables
int pin =0;
int dir = 0;
int level = 0;


//--According to schematic, fill in the table with specific registers of NCT6116D datasheet
//--Activate these GPIO group in Init_GPIO_Controller with LD7 CR30

GPIO_INFO gpo[TOTAL_PINS] = {	
	{ 7, 4, 0xF0, 0xF1, 2},   //DIO_GPO1 : SIO_GP42 -> LD7 CR30[4], Data : F0~F3 bit 2
	{ 7, 4, 0xF0, 0xF1, 1},   //DIO_GPO2 : SIO_GP41 -> LD7 CR30[4], Data : F0~F3 bit 1	
	{ 7, 4, 0xF0, 0xF1, 0},   //DIO_GPO3 : SIO_GP40 -> LD7 CR30[4], Data : F0~F3 bit 0
	{ 7, 3, 0xEC, 0xED, 7},   //DIO_GPO4 : SIO_GP37 -> LD7 CR30[3], Data : EC~EF bit 7
};

GPIO_INFO gpi[TOTAL_PINS] = {	
	{ 7, 3, 0xEC, 0xED, 6},   //DIO_GPI1 : SIO_GP36 -> LD7 CR30[3], Data : EC~EF bit 6
	{ 7, 3, 0xEC, 0xED, 5},   //DIO_GPI2 : SIO_GP35 -> LD7 CR30[3], Data : EC~EF bit 5
	{ 7, 3, 0xEC, 0xED, 4},   //DIO_GPI3 : SIO_GP34 -> LD7 CR30[3], Data : EC~EF bit 4
	{ 7, 3, 0xEC, 0xED, 3},   //DIO_GPI4 : SIO_GP33 -> LD7 CR30[3], Data : EC~EF bit 3
};

//----------------------------------------------------------
void usage( void )
{

	int i = 0;
	const char *Str[]= {             
     "===========================================================",
	 "Options:",
	 "  -h, -?       : Help information." ,
     "  -get n       : Get GPIO PIN n status. n = 01~04 or 11~14" ,
     "  -set n in    : Set GPIO PIN n as input. n = 01~04 or 11~14" , 
	 "  -set n out v : Set GPIO PIN n as output, and value = v (1 or 0).",	
	 "  -all         : Show all DIDO status.",
	 "\nExample:",
	 "  1. Get GPI2 status,",
	 "       MB1_DIDO -get 12",
	 "  2. Set GPO4 to high,",
	 "       MB1_DIDO -set 04 out 1",
	 "  3. Change GPO1 as input pin,",
	 "       MB1_DIDO -set 01 in",
	 "  4. Show all DIDO status",
	 "       MB1_DIDO -all",
     NULL
	};
	
	printf("\n\n %s V%s\n",APP_NAME,APP_VERSION);
	while(Str[i] != NULL) printf("%s\n",Str[i++]);
}

// To configurae SIO, first , get IO permit for the 2E or 4E port.
// Sencond, Unlock the SIO by Enter_Extended_Mode.
// Then User application can access the SIO registers and operate them. (Set_LogicDevice, GetGpioReg, SetGpioReg)
// Finally, Lock again the SIO by Exit_Exteneded_Mode , and free the IO permit for 2E or 4E port. 
//------------------------------------------------------------------------------
void Enter_Extended_Mode (void)
{
   // ioperm(SIO_INDEX, 1, 1);
    //-- To enter extended function mode, 2 successive writes of 0x87 
	//must be applied to Extended Functions Enable Registers(EFER). 
    outportb(SIO_INDEX, 0x87);
	outportb(SIO_INDEX, 0x87);

}
//---------------------------------------------------------------------------
void Exit_Extended_Mode (void)
{
	//-- To exit extended function mode, writeing 0xAA to EFER.
    outportb(SIO_INDEX, 0xAA);
}
//---------------------------------------------------------------------------
void Set_LogicDevice( UINT8 LD)
{	
	//-- To Access the control registers of Logic Devices,
	//Simply write desired logic device number into global register 0x07
    outportb(SIO_INDEX, 0x07 );
    outportb(SIO_DATA, LD);	
}
//------------------------------------------------------------------------
UINT8 GetGpioReg(
    UINT8   LDev,
	UINT8	Register,
	UINT8	bit
)
{
	UINT8	Data8 = 0x00;
	UINT8	Mask;	
	
	Enter_Extended_Mode();
	
	Set_LogicDevice(LDev);
	
	Mask = (1<<bit);
	outportb(SIO_INDEX, Register);
	Data8 = inportb(SIO_DATA);
	Data8 &= Mask;
	Data8 = Data8 >> bit;
	
	Exit_Extended_Mode();
	
	return	Data8;
}
//-------------------------------------------------
void SetGpioReg(
    UINT8   LDev,
	UINT8	Register,
	UINT8	bit,
	UINT8	Value
)
{
	UINT8	Data8 = 0x00;
	UINT8	Mask;
	
	Enter_Extended_Mode();
	
	Set_LogicDevice(LDev);
	
	Mask = (1<<bit);	
	outportb(SIO_INDEX, Register);
	Data8 = inportb(SIO_DATA);
	Data8 &= ~Mask;
	Data8 |= (Value << bit);
	
	outportb(SIO_DATA, Data8);
	
	Exit_Extended_Mode();
}
//---------------------------------------------------------------------------
int Init_GPIO_Controller(void)
{
	int errorcode = MSG_OK;
	UINT8 ucBuf;
    int rc;

    // The caller of ioperm() needs root privilege to get the permission of  
  // IO ports.  	
	rc = ioperm(SIO_INDEX, 2, 1); 
    if ( rc )  { 
    fprintf(stderr, "%s: The caller of ioperm() needs root privilege to get the permission of IO ports.\n",  
      __FUNCTION__ );      
      return MSG_PERMITDENY; 
    }   
  

	//-- Unlock SIO for configuration  
    Enter_Extended_Mode();

    //Enetr Logic Device 7 CR30 to setup GPIO Group 3 & 4.  (CR30[3] & CR30[4])
	Set_LogicDevice(0x07);
	outportb(SIO_INDEX, 0x30);
	ucBuf = inportb(SIO_DATA);
	ucBuf |= (0x01 <<3 | 0x01 <<4 );   //Enable GPIO3, GPIO4 group
	outportb(SIO_INDEX, 0x30);
	outportb(SIO_DATA, ucBuf);

	//-- Lock SIO after configuration  
    Exit_Extended_Mode();
	
    return errorcode;
}

//--------------------------------------------------
void Exit_GPIO_controller(void)
{
  // Release the IO ports 
  ioperm(SIO_INDEX, 2, 0); 
}

//----------------------------------------------------
int Get_GPIO_Status(int pin, int *dir, int *level)
{
	int errorcode = MSG_OK;	
	
	switch(pin)
	{
		case DIO_GPO1:  //ID 01
		case DIO_GPO2:  //ID 02
		case DIO_GPO3:  //ID 03
		case DIO_GPO4:  //ID 04				
		  *dir   = GetGpioReg(gpo[pin-1].LD, gpo[pin-1].REG_DIR, gpo[pin-1].BIT);
		  *level = GetGpioReg(gpo[pin-1].LD, gpo[pin-1].REG_DAT, gpo[pin-1].BIT);
		  break;
		
        case DIO_GPI1:  //ID 11
		case DIO_GPI2:  //ID 12
		case DIO_GPI3:  //ID 13
		case DIO_GPI4:  //ID 14					
		  *dir   = GetGpioReg(gpi[pin-11].LD, gpi[pin-11].REG_DIR, gpi[pin-11].BIT);
		  *level = GetGpioReg(gpi[pin-11].LD, gpi[pin-11].REG_DAT, gpi[pin-11].BIT);
		  break;


		default:  
		    printf("Invalid GPIO pin!") ;
		    errorcode = MSG_InvalidParam;
		    break;
			 
	}	
	
	return errorcode;
}

//---------------------------------------------------
int Set_GPIO_Status(int pin, int dir, int level)
{
	int errorcode = MSG_OK;
	
	//printf("dir:%d  level:%d\n",dir,level);
	switch(pin)
	{
		case DIO_GPO1:  //ID 01
		case DIO_GPO2:  //ID 02
		case DIO_GPO3:  //ID 03
		case DIO_GPO4:  //ID 04	
			SetGpioReg(gpo[pin-1].LD, gpo[pin-1].REG_DIR, gpo[pin-1].BIT, dir);
			SetGpioReg(gpo[pin-1].LD, gpo[pin-1].REG_DAT, gpo[pin-1].BIT, level);		
			break;

        case DIO_GPI1:  //ID 11
		case DIO_GPI2:  //ID 12
		case DIO_GPI3:  //ID 13
		case DIO_GPI4:  //ID 14	
			SetGpioReg(gpi[pin-11].LD, gpi[pin-11].REG_DIR, gpi[pin-11].BIT, dir);
			SetGpioReg(gpi[pin-11].LD, gpi[pin-11].REG_DAT, gpi[pin-11].BIT, level);
			break;
		   
		default:  
		    printf("Invalid GPIO pin!") ;
		    errorcode = MSG_InvalidParam;
		    break;
			 
	}	
	
	return errorcode;
}
//---------------------------------------------------------
int Show_BoardStatus(void)
{
   int errorcode = MSG_OK;
  
   char bits_dir[8], bits_lev[8];  

   //-- The GPO 1~4
   for(int i = 0; i < 4; i++ )
   {
	   int dir=0, level=0;
	   int err = Get_GPIO_Status( i+1, &dir, &level);
	   if(err) 
	   {
		   bits_dir[i] = 'E';
		   bits_lev[i] = 'E';
	   }
	   else
	   {
		   bits_dir[i] = (dir) ? 'I' : 'O';
		   bits_lev[i] = (level) ? '1': '0';
	   }
   }

   //-- The GPI 1~4
   for(int i = 0; i < 4; i++ )
   {
	   int dir=0, level=0;
	   int err = Get_GPIO_Status( i+11, &dir, &level);
	   if(err) 
	   {
		   bits_dir[i+4] = 'E';
		   bits_lev[i+4] = 'E';
	   }
	   else
	   {
		   bits_dir[i+4] = (dir) ? 'I' : 'O';
		   bits_lev[i+4] = (level) ? '1': '0';
	   }
   }


   printf(" Borad DIO Status :\n");
   printf("             OOOO-IIII \n");
   printf("      DIO    1234-1234 \n");
   printf("             ==== ==== \n");
   printf("     In/Out: ");   
   for(int i = 0; i < 8; i++)
	{
		printf("%c", bits_dir[i]);
		if(((i+1) % 4) == 0) printf(" ");
	}  
   printf("\n     Level : ");
   for(int i = 0; i < 8; i++)
	{
		printf("%c", bits_lev[i]);
		if(((i+1) % 4) == 0) printf(" ");
	}  
   printf("\n");

   return errorcode;
}

//---
unsigned int GetChipID(unsigned short adr)
{
	unsigned int uuid;


//	printf("CR26h=%02X\n", inportb(0x26) );

	//--Enter Extention mode
	outportb(adr, 0x87);
	outportb(adr, 0x87);

//	printf("CR26h=%02X\n", inportb(0x26) );
	//--
	outportb(adr, 0x20);
	uuid = (inportb(adr+1)) << 8;
	outportb(adr, 0x21);
	uuid |= (inportb(adr+1));
	
	//--Exit extension mode
	outportb(adr, 0xAA);

	return uuid;
}

//--------------------------------------------------------------------
int main(int argc, char* argv[])
{
	int errorcode = MSG_OK;

    //---- Paramenters handle... 
	if (argc < 2)
	{
		usage();
		errorcode = MSG_HELP;
		goto ending;
	}

	if ((strcmp(argv[1], "-?") == 0) || (strcmp(argv[1], "-h") == 0))
	{
		usage();
		errorcode = MSG_HELP;
		goto ending;
	}

//---------------------------------------
	//os = OsInit();

	 //-- Initialize Super IO 	
  if(Init_GPIO_Controller() != MSG_OK)
    {	  
	  return MSG_PERMITDENY;
    }
	
   //-- Main functions start here 	
  if(strcmp(argv[1],"-get") == 0 )
    {
       if(argc == 3)
	   {       	
       	  pin = atoi(argv[2]) ;	
       	  errorcode = Get_GPIO_Status(pin, &dir, &level);
       	  if(errorcode == MSG_OK) printf("GP%02d : %s  %s\n",pin, dir?"Input(1)":"Output(0)", level?"High(1)":"Low(0)");       	  
	   } 
	   else 
	   {
	   	  printf("Invalid parameters! use -h or -? to get help. \n");
	   	  errorcode = MSG_InvalidParam;
	   }	              
    }
  else if(strcmp(argv[1],"-set") == 0 )
    {
        if(argc > 3) pin = atoi(argv[2]) ;
        else {
	   	  printf("Invalid parameters! use -h or -? to get help. \n");
	   	  errorcode =  MSG_InvalidParam;
	    }
	   
        if(strcmp(argv[3], "in") == 0)
        {
        	dir = 1;
			level = 0; // This level value is dummy since the GPIO is set as input mode.  
			errorcode = Set_GPIO_Status(pin, dir, level);			
       	  //  if(errorcode == MSG_OK) printf("DIO_%02d is set as Input pin.\n",pin);       	   
		}
		else if (strcmp(argv[3], "out") == 0)
		{
			dir = 0;
			level = (atoi(argv[4])>0)?1:0;  
			errorcode = Set_GPIO_Status(pin, dir, level);			
       	  //  if(errorcode == MSG_OK) printf("DIO_%02d is set as Output pin with value = %d.\n", pin, level);       	    
		}       
    }
  else if(strcmp(argv[1],"-all") == 0 )
    {
       errorcode = Show_BoardStatus();		
    }
  else if(strcmp(argv[1],"-test") == 0 )
    {
       unsigned int uid;
	   uid = GetChipID(0x2E);
	   printf("ID(0x2E)= %04X\n", uid);
	   uid = GetChipID(0x4E);
	   printf("ID(0x4E)= %04X\n", uid);
    }
  else
    {
       printf("Unknown argument! use -h or -? to get help.\n");       
       return MSG_HELP;
    }      



	Exit_GPIO_controller();
//--------------------------------------
ending:
	return errorcode;
}

